<xsl:stylesheet version="2.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform" xmlns:cfg="." xmlns:msxsl="urn:schemas-microsoft-com:xslt" xmlns:user2="http://user2.bosch.com">
  <xsl:output method="xml" encoding="UTF-8" indent="yes"/>
  <msxsl:script language="javascript" implements-prefix="user2">
	<![CDATA[
/*	
Copyright (C) 1998, 2009
Paul E. Jones <paulej@packetizer.com>

Freeware Public License (FPL)

This software is licensed as "freeware."  Permission to distribute
this software in source and binary forms, including incorporation 
into other products, is hereby granted without a fee.  THIS SOFTWARE 
IS PROVIDED 'AS IS' AND WITHOUT ANY EXPRESSED OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
AND FITNESS FOR A PARTICULAR PURPOSE.  THE AUTHOR SHALL NOT BE HELD 
LIABLE FOR ANY DAMAGES RESULTING FROM THE USE OF THIS SOFTWARE, EITHER 
DIRECTLY OR INDIRECTLY, INCLUDING, BUT NOT LIMITED TO, LOSS OF DATA 
OR DATA BEING RENDERED INACCURATE.
*/	
/*
 *  sha1.h
 *
 *  Copyright (C) 1998, 2009
 *  Paul E. Jones <paulej@packetizer.com>
 *  All Rights Reserved
 *
 *****************************************************************************
 *  $Id: XSLT/hashFunction.xsl 1.3 2015/05/18 11:27:53GMT Maria Lux (lux2ot) Exp  $
 *****************************************************************************
 *
 *  Description:
 *      This class implements the Secure Hashing Standard as defined
 *      in FIPS PUB 180-1 published April 17, 1995.
 *
 *      Many of the variable names in the SHA1Context, especially the
 *      single character names, were used because those were the names
 *      used in the publication.
 *
 *      Please read the file sha1.c for more information.
 *
 */
/*
 *  sha1.c
 *
 *  Copyright (C) 1998, 2009
 *  Paul E. Jones <paulej@packetizer.com>
 *  All Rights Reserved
 *
 *****************************************************************************
 *  $Id: XSLT/hashFunction.xsl 1.3 2015/05/18 11:27:53GMT Maria Lux (lux2ot) Exp  $
 *****************************************************************************
 *
 *  Description:
 *      This file implements the Secure Hashing Standard as defined
 *      in FIPS PUB 180-1 published April 17, 1995.
 *
 *      The Secure Hashing Standard, which uses the Secure Hashing
 *      Algorithm (SHA), produces a 160-bit message digest for a
 *      given data stream.  In theory, it is highly improbable that
 *      two messages will produce the same message digest.  Therefore,
 *      this algorithm can serve as a means of providing a "fingerprint"
 *      for a message.
 *
 *  Portability Issues:
 *      SHA-1 is defined in terms of 32-bit "words".  This code was
 *      written with the expectation that the processor has at least
 *      a 32-bit machine word size.  If the machine word size is larger,
 *      the code should still function properly.  One caveat to that
 *      is that the input functions taking characters and character
 *      arrays assume that only 8 bits of information are stored in each
 *      character.
 *
 *  Caveats:
 *      SHA-1 is designed to work with messages less than 2^64 bits
 *      long. Although SHA-1 allows a message digest to be generated for
 *      messages of any number of bits less than 2^64, this
 *      implementation only works with messages with a length that is a
 *      multiple of the size of an 8-bit character.
 *
 */

 /*  
*  This structure will hold context information for the hashing operation 
*/
	var Message_Digest; /* Message Digest (output)          */    
	var Length_Low;        /* Message length in bits           */    
	var Length_High;       /* Message length in bits           */    
	var Message_Block; /* 512-bit message blocks      */    
	var Message_Block_Index;    /* Index into message block array   */    
	var Computed;               /* Is the digest computed?          */    
	var Corrupted;              /* Is the message digest corruped?  */

function Normalize(word)
{
	return (word & 0xFFFFFFFF);
}

/*   *  SHA1Reset 
* 
*  Description: 
*      This function will initialize the SHA1Context in preparation 
*      for computing a new message digest. 
* 
*  Parameters: 
*      sha1Context: [in/out] 
*          The context to reset. 
* 
*  Returns: 
*      Nothing. 
* 
*  Comments: 
* 
*/
function SHA1Reset()
{
   Length_Low             = 0;    
   Length_High            = 0;    
   Message_Block_Index    = 0;    
   Message_Digest= [0x67452301, 0xEFCDAB89, 0x98BADCFE, 0x10325476, 0xC3D2E1F0];    
   Message_Block = new Array(64);
   Computed   = false;    
   Corrupted  = false;
}

/*  
 *  SHA1Result
 *
 *  Description:
 *      This function will return the 160-bit message digest into the
 *      Message_Digest array within the SHA1Context provided
 *
 *  Parameters:
 *      sha1Context: [in/out]
 *          The context to use to calculate the SHA-1 hash.
 *
 *  Returns:
 *      1 if successful, 0 if it failed.
 *
 *  Comments:
 *
 */
function SHA1Result()
{

    if (Corrupted)
    {
        return 0;
    }

    if (!Computed)
    {
        SHA1PadMessage();
        Computed = true;
    }

    return true;
}

/*  
 *  SHA1Input
 *
 *  Description:
 *      This function accepts an array of octets as the next portion of
 *      the message.
 *
 *  Parameters:
 *      sha1Context: [in/out]
 *          The SHA-1 context to update
 *      message_array: [in]
 *          An array of characters representing the next portion of the
 *          message.
 *      length: [in]
 *          The length of the message in message_array
 *
 *  Returns:
 *      Nothing.
 *
 *  Comments:
 *
 */
function SHA1Input( message_array,length)
{
    if (length == 0)
    {
        return;
    }

    if (Computed || Corrupted)
    {
        Corrupted = true;
        return;
    }

	var index = 0;
    while((index < length) && !Corrupted)
    {
        Message_Block[Message_Block_Index++] = (message_array[index] & 0xFF);

        Length_Low += 8;
        /* Force it to 32 bits */
        Length_Low &= 0xFFFFFFFF;
        if (Length_Low == 0)
        {
            Length_High++;
            /* Force it to 32 bits */
            Length_High &= 0xFFFFFFFF;
            if (Length_High == 0)
            {
                /* Message is too long */
                Corrupted = true;
            }
        }

        if (Message_Block_Index == 64)
        {
            SHA1ProcessMessageBlock();
        }

        index++;
    }
}

/* original c-define
#define SHA1CircularShift(bits,word) \                ((((word) << (bits)) & 0xFFFFFFFF) | \                ((word) >> (32-(bits))))
*/

function MyShiftRight(bits, word)
{
	var bIsNegative = false;
	if ((word < 0) || (word >= 0x80000000))
	{
		bIsNegative = true;
	}
	var retValue = (word & 0x7FFFFFFF) >> bits;
	if (bIsNegative)
	{
		// add the missing bit
		retValue |= (1 << (31 - bits));
	}
	return retValue
}
function SHA1CircularShift(bits,word)
{               
	var retValue = ((word <<bits) & 0xFFFFFFFF) |   (MyShiftRight(32-bits, word));
	return retValue;
}
/*  
 *  SHA1ProcessMessageBlock
 *
 *  Description:
 *      This function will process the next 512 bits of the message
 *      stored in the Message_Block array.
 *
 *  Parameters:
 *      None.
 *
 *  Returns:
 *      Nothing.
 *
 *  Comments:
 *      Many of the variable names in the SHAContext, especially the
 *      single character names, were used because those were the names
 *      used in the publication.
 *         
 *
 */
function SHA1ProcessMessageBlock()
{
	/* Constants defined in SHA-1   */  
    var K =  [ 0x5A827999, 0x6ED9EBA1, 0x8F1BBCDC, 0xCA62C1D6];
    
    var         t;                  /* Loop counter                 */
    var    temp;               /* Temporary word value         */
    var    W = new Array(80);              /* Word sequence                */
    var    A, B, C, D, E;      /* Word buffers                 */

    /*
     *  Initialize the first 16 words in the array W
     */
    for(t = 0; t < 16; t++)
    {
        W[t] = ( Message_Block[t * 4]) << 24;
        W[t] |= (Message_Block[t * 4 + 1]) << 16;
        W[t] |= (Message_Block[t * 4 + 2]) << 8;
        W[t] |= (Message_Block[t * 4 + 3]);
		W[t] &= 0xFFFFFFFF;	// always make sure it does not exceed 32 bits
    }

    for(t = 16; t < 80; t++)
    {
		temp = W[t-3] ^ W[t-8] ^ W[t-14] ^ W[t-16];
        temp &= 0xFFFFFFFF;
       W[t] = SHA1CircularShift(1,temp);
		W[t] &= 0xFFFFFFFF;	// always make sure it does not exceed 32 bits
    }

    A = Message_Digest[0];
    B = Message_Digest[1];
    C = Message_Digest[2];
    D = Message_Digest[3];
    E = Message_Digest[4];

    for(t = 0; t < 20; t++)
    {
		var temp1 = SHA1CircularShift(5,A);
		temp1 = Normalize(temp1);
		var temp2 = (B & C);
		temp2 = Normalize(temp2);
		var temp3 = ((~B) & D);
		temp3 = Normalize(temp3);
		var temp4 = temp2 | temp3;
		temp4 = Normalize(temp4);
		var temp5 = E + W[t] + K[0];
		temp5 = Normalize(temp5);
		temp = temp1 + temp4 + temp5;
		temp = Normalize(temp);
        E = D;
        D = C;
        C = SHA1CircularShift(30,B);
		C &= 0xFFFFFFFF;	// always make sure it does not exceed 32 bits
        B = A;
        A = temp;
    }

    for(t = 20; t < 40; t++)
    {
	
		var temp1 = SHA1CircularShift(5,A);
		temp1 = Normalize(temp1);
		var temp2 = (B ^ C ^ D);
		temp2 = Normalize(temp2);
		var temp3 = E + W[t] + K[1];
		temp3 = Normalize(temp3);
		temp = temp1 + temp2 + temp3;
		temp = Normalize(temp);
        E = D;
        D = C;
        C = SHA1CircularShift(30,B);
		C &= 0xFFFFFFFF;	// always make sure it does not exceed 32 bits
        B = A;
        A = temp;
    }

    for(t = 40; t < 60; t++)
    {
        //temp = SHA1CircularShift(5,A) +
        //       ((B & C) | (B & D) | (C & D)) + E + W[t] + K[2];
        //temp &= 0xFFFFFFFF;

		var temp1 = SHA1CircularShift(5,A);
		temp1 = Normalize(temp1);
		var temp2 = (B & C);
		temp2 = Normalize(temp2);
		var temp3 = (B & D);
		temp3 = Normalize(temp3);
		var temp4 =  (C & D);
		temp4 = Normalize(temp4);
		var temp5 = temp2 | temp3 | temp4;
		temp5 = Normalize(temp5);
		var temp6 = E + W[t] + K[2];
		temp6 = Normalize(temp6);
		temp = temp1 + temp5 + temp6;
		temp = Normalize(temp);
        E = D;
        D = C;
        C = SHA1CircularShift(30,B);
		C &= 0xFFFFFFFF;	// always make sure it does not exceed 32 bits
        B = A;
        A = temp;
    }

    for(t = 60; t < 80; t++)
    {
        //temp = SHA1CircularShift(5,A) + (B ^ C ^ D) + E + W[t] + K[3];
        //temp &= 0xFFFFFFFF;
		var temp1 = SHA1CircularShift(5,A);
		temp1 = Normalize(temp1);
		var temp2 = (B ^ C ^ D);
		temp2 = Normalize(temp2);
		var temp3 = E + W[t] + K[3];
		temp3 = Normalize(temp3);
		temp = temp1 + temp2 + temp3;
		temp = Normalize(temp);
        E = D;
        D = C;
        C = SHA1CircularShift(30,B);
		C &= 0xFFFFFFFF;	// always make sure it does not exceed 32 bits
        B = A;
        A = temp;
    }

    Message_Digest[0] =
                        (Message_Digest[0] + A) & 0xFFFFFFFF;
    Message_Digest[1] =
                        (Message_Digest[1] + B) & 0xFFFFFFFF;
    Message_Digest[2] =
                        (Message_Digest[2] + C) & 0xFFFFFFFF;
    Message_Digest[3] =
                        (Message_Digest[3] + D) & 0xFFFFFFFF;
    Message_Digest[4] =
                        (Message_Digest[4] + E) & 0xFFFFFFFF;

    Message_Block_Index = 0;
}

/*  
 *  SHA1PadMessage
 *
 *  Description:
 *      According to the standard, the message must be padded to an even
 *      512 bits.  The first padding bit must be a '1'.  The last 64
 *      bits represent the length of the original message.  All bits in
 *      between should be 0.  This function will pad the message
 *      according to those rules by filling the Message_Block array
 *      accordingly.  It will also call SHA1ProcessMessageBlock()
 *      appropriately.  When it returns, it can be assumed that the
 *      message digest has been computed.
 *
 *  Parameters:
 *      sha1Context: [in/out]
 *          The context to pad
 *
 *  Returns:
 *      Nothing.
 *
 *  Comments:
 *
 */
function SHA1PadMessage()
{
    /*
     *  Check to see if the current message block is too small to hold
     *  the initial padding bits and length.  If so, we will pad the
     *  block, process it, and then continue padding into a second
     *  block.
     */
    if (Message_Block_Index > 55)
    {
        Message_Block[Message_Block_Index++] = 0x80;
        while(Message_Block_Index < 64)
        {
            Message_Block[Message_Block_Index++] = 0;
        }

        SHA1ProcessMessageBlock();

        while(Message_Block_Index < 56)
        {
            Message_Block[Message_Block_Index++] = 0;
        }
    }
    else
    {
        Message_Block[Message_Block_Index++] = 0x80;
        while(Message_Block_Index < 56)
        {
            Message_Block[Message_Block_Index++] = 0;
        }
    }

    /*
     *  Store the message length as the last 8 octets
     */
    Message_Block[56] = MyShiftRight(24, Length_High) & 0xFF;
    Message_Block[57] = MyShiftRight(16, Length_High) & 0xFF;
    Message_Block[58] = MyShiftRight(8, Length_High) & 0xFF;
    Message_Block[59] = Length_High & 0xFF;
    Message_Block[60] = MyShiftRight(24, Length_Low) & 0xFF;
    Message_Block[61] = MyShiftRight(16, Length_Low) & 0xFF;
    Message_Block[62] = MyShiftRight(8, Length_Low) & 0xFF;
    Message_Block[63] = Length_Low & 0xFF;

    SHA1ProcessMessageBlock();
}
function hex2a(hexx) {
	var str = "";
	var byteArray = new Array(8);
	byteArray[0] = MyShiftRight(28, hexx) & 0x0F;
	byteArray[1] = MyShiftRight(24, hexx) & 0x0F;
	byteArray[2] = MyShiftRight(20, hexx) & 0x0F;
	byteArray[3] = MyShiftRight(16, hexx) & 0x0F;
	byteArray[4] = MyShiftRight(12, hexx) & 0x0F;
	byteArray[5] = MyShiftRight(8, hexx) & 0x0F;
	byteArray[6] = MyShiftRight(4, hexx) & 0x0F;
	byteArray[7] = hexx  & 0x0F;
	var toDo = 8;
	while (toDo)
	{
		var hexDigits = ["0", "1", "2", "3", "4", "5", "6", "7", "8", "9", "A", "B", "C", "D", "E", "F"];
		var currentByte = byteArray[8-toDo];
		str += hexDigits[currentByte];
		toDo--;
	}
    return str;
}

function HashPassword(inPassword)
{
	var salt = [0x5E, 0xC0, 0xA3, 0x2A, 0x69, 0x38, 0x2C, 0xa9];
	var outPassword = "4A154B2B";
	SHA1Reset();
	var inPasswordArray = new Array(inPassword.length);
	for (var i = 0; i < inPassword.length; i++)
	{
		inPasswordArray[i] = inPassword.charCodeAt(i);
	}
	SHA1Input(inPasswordArray, inPassword.length);
	SHA1Input(salt, 8);
	if (SHA1Result() == true)
	{
		outPassword = hex2a(Message_Digest[0]);
	}
	return outPassword;
}

]]>
</msxsl:script>  
</xsl:stylesheet>